import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import HandwrittenCategoryClient from "./HandwrittenCategoryClient";
import { buildAssignmentCourseMeta } from "@/lib/seo/assignmentMeta";

export const dynamic = "force-dynamic";

async function fetchHandwrittenCourseData(courseCode: string) {
    try {
        const res = await API_ENDPOINTS.ASSIGNMENTS.COURSE_DETAIL('handwritten', courseCode);
        if (!res.ok) throw new Error(`HTTP ${res.status}`);

        const data = await res.json();
        return {
            course: data.course,
            list: data.list || []
        };
    } catch (err: any) {
        console.error("❌ Failed to load handwritten course details", err);
        return null;
    }
}

export async function generateMetadata({ params }: { params: Promise<{ slug: string }> }): Promise<Metadata> {
    const resolvedParams = await params;
    const data = await fetchHandwrittenCourseData(resolvedParams.slug);
    if (!data || !data.course) return { title: "Handwritten Course Assignments" };

    const meta = buildAssignmentCourseMeta(data.course);
    return {
        ...meta,
        alternates: {
            canonical: `/assignments/handwritten/${resolvedParams.slug}`,
        }
    };
}

export default async function HandwrittenCoursePage({ params }: { params: Promise<{ slug: string }> }) {
    const resolvedParams = await params;
    const data = await fetchHandwrittenCourseData(resolvedParams.slug);

    if (!data) {
        return <div className="text-danger text-center py-5">Failed to load course details.</div>;
    }

    return (
        <HandwrittenCategoryClient
            courseData={data.course}
            assignmentsList={data.list}
            courseCode={resolvedParams.slug}
        />
    );
}
